/*:::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::
::
::   Module      :   USB Resource Framework API Header File
::   Copyright   :   (C)2009 Woodward
::   Platform(s) :   MPC5xx
::   Dependants  :   USB stack
::
:::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::*/
/*! \file Resource_USB.h
    \brief This attaches to a USB stack implementation for both HOST and PERIPHERAL functionality with
    selected class drivers.
*/

#ifndef __RESOURCE_USB_H
#define __RESOURCE_USB_H

/*----- INCLUDES ------------------------------------------------------------------------------------------*/
#include <typedefn.h>
#include <MacroDefn.h>
#include <resource.h>

/*----- DEFINES -------------------------------------------------------------------------------------------*/

/*----- TYPEDEFS ------------------------------------------------------------------------------------------*/
#pragma pack(1)

/*! \brief Enumeration describes the set of mask definitions that are used to identify the valid USB
           attributes in the S_USBCreateAttributes data structure.

Each mask definition applies to a particular member of one of the data structures.
\code
    ...
    S_USBCreateAttributes AttribsObj;

    AttribsObj.uValidAttributesMask = USE_USB_HOST;
    AttribsObj.uHostClassDriverMask = CLASSDRIVER_MASSSTORAGE;
    ... \endcode */
typedef enum
{
/*! Include this mask in S_USBCreateAttributes::uValidAttributesMask to enable the USB host functionality on the stack. */
    USE_USB_HOST = 0x00000001UL,
/*! Include this mask in S_USBCreateAttributes::uValidAttributesMask to enable host class drivers given by
    S_USBCreateAttributes::uHostClassDriverMask */
    USE_USB_HOSTCLASSDRIVER = 0x00000002UL,

/*! Include this mask in S_USBCreateAttributes::uValidAttributesMask when the application desires to only support the specific
    set of mass storage class drivers defined by S_USBCreateAttributes::uHostClassMSDriverMask */
    USE_USB_HOSTCLASS_MS_SPECIFIC_DEVICE = 0x00000004UL,

/*! Include this mask in S_USBCreateAttributes::uValidAttributesMask to enable the USB peripheral functionality on the stack. */
    USE_USB_PERIPHERAL = 0x00010000UL,
/*! Include this mask in S_USBCreateAttributes::uValidAttributesMask to enable peripheral class drivers given by
    S_USBCreateAttributes::uPeripheralClassDriverMask */
    USE_USB_PERIPHERALCLASSDRIVER = 0x00020000UL,

/*! Include this mask in S_USBCreateAttributes::uValidAttributesMask to enable the stack debug trace on the port given by
    S_USBCreateAttributes::eDebugPort. */
    ENABLE_DEBUG_TRACE = 0x40000000UL,
} E_USBCreateAttributeMask;

typedef uint4 USBCreateAttributesMask_U;

/*! List of supported class drivers. */
typedef enum
{
/*! Include this mask in S_USBCreateAttributes::uHostClassDriverMask to enable mouse support on the host stack, 
    and/or S_USBCreateAttributes::uPeripheralClassDriverMask to enable mouse support on the peripheral stack */
    CLASSDRIVER_MOUSE = 0x01,
/*! Include this mask in S_USBCreateAttributes::uHostClassDriverMask to enable mass storage on the host stack, 
    and/or S_USBCreateAttributes::uPeripheralClassDriverMask to enable mass storage on the peripheral stack */
    CLASSDRIVER_MASS_STORAGE = 0x02,
/*! Include this mask in S_USBCreateAttributes::uHostClassDriverMask to enable factory test on the host stack, 
    and/or S_USBCreateAttributes::uPeripheralClassDriverMask to enable factory test on the peripheral stack */
    CLASSDRIVER_FACTORY_TEST = 0x04,
} E_USBClassDriverMask;

typedef uint1 USBClassDriverMask_U;

/*! Describes the specific mass storage devices that can be separately identified by the framework */
typedef enum
{
    USB_MS_DEVICE_ANY           = 0x01,     /*!< Any mass storage device */
    USB_MS_DEVICE_LEXAR_ANY     = 0x02,     /*!< Any Lexar mass storage device */
    USB_MS_DEVICE_BELKIN_ANY    = 0x04,     /*!< Any Belkin mass storage device */
    USB_MS_DEVICE_USB007_256MB  = 0x08,     /*!< USB007 256MByte device */
} E_USBClassMSDriverMask;

typedef uint1 USBMSClassDriverMask_U;

/*! Describes the creation attributes for a resource with the \ref BEHAVIOUR_USB */
typedef struct
{
/*! Logic-OR the attributes [\ref USE_USB_HOST etc] that are valid for this instance of the data structure. 
    The framework will employ the creation defaults for those attributes that are left undefined. */
    USBCreateAttributesMask_U uValidAttributesMask;
/*! Logic-OR the attributes [\ref CLASSDRIVER_MOUSE etc] to attach the desired class drivers to the host stack. */
    USBClassDriverMask_U uHostClassDriverMask;
/*! Logic-OR the attributes [\ref CLASSDRIVER_MOUSE etc] to attach the desired class drivers to the peripheral stack. */
    USBClassDriverMask_U uPeripheralClassDriverMask;
/*! The \ref BEHAVIOUR_COMM compatible serial resource to use as a debug trace port when [\ref ENABLE_DEBUG_TRACE]
    is enabled. */
    E_ModuleResource eDebugPort;
/*! The baud rate to use for the debug trace port when [\ref ENABLE_DEBUG_TRACE] is enabled. */
    uint4 uDebugBaudRate;
/*! Logic-OR the desired set of mass storage devices that are to be supported when the mass storage class driver is
    instantiated. Select this attribute with the \ref USE_USB_HOSTCLASS_MS_SPECIFIC_DEVICE. If uninitialised then
    this member defaults to USB_MS_DEVICE_ANY */
    USBMSClassDriverMask_U uHostClassMSDriverMask;
} S_USBCreateAttributes;

#pragma pack()

/*----- EXTERNALS -----------------------------------------------------------------------------------------*/

/*----- PROTOTYPES ----------------------------------------------------------------------------------------*/
NativeError_S CreateResourceBEHAVIOUR_USB(E_ModuleResource in_eResource, S_USBCreateAttributes const* in_pAttribObj);
NativeError_S DestroyResourceBEHAVIOUR_USB(E_ModuleResource in_eResource);

#endif /* __RESOURCE_USB_H */

/*----- END OF FILE ---------------------------------------------------------------------------------------*/
